classdef myOutputFile
    methods (Static)
        %% OutputFile - (file setup) Creazione del file*.txt con un nome specificato in input ed inserimento degli headers di premessa
        function fileID = preparaFile(fileName,headersList)
            %(?) cosa fa: data una stringa (fileName) contenente il full
            %path di un file di testo (che si intende creare) e un array di stringhe (headerList),
            %questa function crea il file e scrive nella prima riga gli
            %header separati tra loro dal 'tab'. Restituisce in uscita il
            %puntatore al file, senza chiuderlo (poiché sicuramente servirà
            %scriverci ancora dentro).
            %(^) Syntax:
            %  f = preparaFile( 'C:\ ..fullPath..\myFile.txt', ["time" "Mic1" "Mic2"])
            fileID = fopen(fileName,'w');
            titoli = strjoin(string(headersList),'\t');
            fprintf(fileID,[char(titoli) '\n']);
        end

        %% OutputFile - (file naming convention) Richiesta all'utente su come chiamare i file per la misurazione.
        function [structFileName,channelValid,ora,oraStr] = setFileName( mainDir, subFolders, enableChan, varargin)
        %(?) cosa fa: chiede all'utente come intende impostare il nome
        %standard dei file di simulazione, e, in base ai devices
        %disponibili e alle intenzioni dell'utente, declina poi il nome
        %standard in tutte le sue varianti in base ai dispositivi,
        %assegnando inoltre un timestamp temporale (data+ora) per impedire
        %conflitti con simulazioni omonime.
        %(>) lista input:
        %  1-  mainDir : directory di base (fullpath), di solito meglio la pwd.
        %  2-  subFolder : vettore di stringhe, di default dovrebbe essere in quest'ordine ["Loa" "Mic" "Get" "Note"]
        %       queste keyword trasformeranno il baseName dell'utente in
        %       versioni specifiche, e saranno le properties della struct
        %       in uscita, ammesso che la connessione sia possibile.
        %  3-  enableChan : vettore di booleani [1 x 3] dove l'utente dice
        %      quali devices vuole utilizzare (Load cell, Mics, Get)
        %  4°,5°,6°- varargin: rispettivamente, le variabili dq, dm, dg, in modo che la 
        %      setFileName possa verificare che il device sia correttamente
        %      configurato.
        %(^) Syntax:
        %    [dq,dm,dg] = connect2DAQ(true, true)
        %    [s, c, orario, orarioString ] = setFileName(  pwd, ["Loa" "Mic" "Get" "Note"], [1 1 1], dq, dm, dg) 
        %    -> si aprirà un'interfaccia, l'utente dovrà impostare un nome e poi usciranno questi output:
        % s.Loa = 'C:\..pwd..\userFile TEST_Loa (data ora).txt'
        % s.Mic = 'C:\..pwd..\userFile TEST_Mic (data ora).txt'
        % s.Get = 'C:\..pwd..\userFile TEST_Get (data ora).txt'
        % s.Note = 'C:\..pwd..\userFile TEST_Note (data ora).txt'
        %
        % c = [1 1 1] un vettore di booleani in base alla disponibilità di dispositivi DAQ realmente attivi
        %             in relazione ai desideri dell'utente.
        % 
        % orario = orario (dateVec) di riferimento di futura creazione file
   
        global ax
            permission = ~isfolder(mainDir);

            % Si verifica che il file sia effettivamente connesso alla
            % macchina: se possiede dei 'Channels' allora è realmente
            % connesso.
            channelVec = zeros(1,3);
            for i=1:length(varargin)
               channelVec(i) = numel(varargin{i}.Channels);  %<-conto il numero di channels
            end

            % inizializza baseName, ovvero un possibile nome di default, e memorizza l'orario come timestamp
            structFileName = struct();
            consigliato = ['TEST' 'X '];
            ora = now;
            oraStr = datestr(ora,'(mmm dd HH_MM_SS)');

            % se l'interfaccia è attiva, mostra l'orario di registrazione
            if isvalid(ax{4})
                ax{4}.Title.String = "New Session - "+replace(oraStr,'_',':');
            end
            fprintf("New Session - "+oraStr);


            if permission
            % se non esiste una cartella di destinazione, propone la directory principale,
            % e chiede all'utente di dichiarare il nome generico della
            % simulazione che intende utilizzare.
            % Poi inserisce la 'X' che servirà come placeholder sostitutivo per le tag
            % (_Loa, _Mics, _Get, _Note)
                mainDir = pwd;
               consigliato = [mainDir filesep consigliato oraStr '.txt'];
               [a,b]=uiputfile(consigliato,'Con che nome vuoi salvare?   ___TESTX (data ora).txt');
               a = string(a); b=string(b);
               if ~contains(a,'X')
                a = string(replace(a,'.txt','X.txt'));
               end
            else
            % se invece esiste una cartella di destinazione, quella diviene la directory,
            % e l'utente non può scegliere il nome del file, poiché (si
            % suppone) sia una esecuzione da svolgere in automatico per più volte
            % All'interno del nome consigliato è presente sempre il placeholder X
            % che servirà per declinare il baseName file nei suoi
            % sottoposti.
               a=string(consigliato)+oraStr+'.txt';
               b=string(mainDir);
              if ~endsWith(b,filesep)
                b = b+filesep;     
              end
            end

            % Si verifica quali tra ["Loa", "Mic", "Get"]=subFolders siano da salvare.
            % Bisogna verificare per ognuno ben 2 condizioni:
            % -che il canale sia realmente connesso e configurato:(enableChan)
            % -che l'utente abbia intenzione di utilizzarlo:(channelVec)
            channelValid = false(1,3);
            for i=1:length(subFolders)
                if i<=length(channelVec)
                 if enableChan(i) && channelVec(i)
                  channelValid(i) = true; %<--il dispositivo Loa/Mic/Get ora è considerato operativo.
                  % viene creata la property associata. 
                  % per esempio, se il file si chiama 'myFile TESTX.txt'
                  % si ottiene:
                  % >> baseName.Loa = 'C:\..fullPath..\myFile TEST_Loa.txt'
                   % >> baseName.Mic = 'C:\..fullPath..\myFile TEST_Mic.txt'
                    % >> baseName.Get = 'C:\..fullPath..\myFile TEST_Get.txt'
                  structFileName.(subFolders(i)) = b+replace(a,'X','_'+subFolders(i));    
                 end
                else
                 structFileName.(subFolders(i)) = b+replace(a,'X','_'+subFolders(i));   
                end
            end
            % per spostare poi tali file sulle sottocartelle Loa/Mic/Get si
            % utilizzerà la function migraFile, che si attiva solo dopo che
            % la simulazione si conclude con successo.
        end

        %% OutputFile - (folder check-up) Verifica delle directories, ed eventuale creazione
        function SetUpFolders(mainDir, sessionFolder, subFolders)
        %(?) cosa fa: quando l'utente intende procedere con una nuova
        %sessione di registrazioni, solitamente preferirebbe salvare tutti
        %i risultati in una nuova cartella, per non mischiare i files con
        %altre sessioni. Ebbene, SetUpFolders( mainDir, sessionFolder, subFolders)
        % controlla che le cartelle per la nuova sessione esistano, e, in
        % caso contrario, le crea. Inoltre, aggiunge tutte le funzionalità
        % presenti nella cartella di progetto, in modo da mettere in
        % comunicazione (e poter utilizzare) tutte le librerie di functions.
        %(>) input list:
        %  1- mainDir : è il percorso completo di progetto, per esempio:
        %    ('C:\..full..\mainDir\')  <-- in cui dovrebbero trovarsi tutte
        %    le librerie.
        %  2- sessionFolder : è il nome della nuova cartella che si intende
        %    agganciare a mainDir, per esempio: 'test di Martedì 17'
        %  3- subFolders : un vettore di stringhe con tutte le
        %     sottocartelle da creare dentro a sessionFolder, di solito questo
        %     è = ["Loa", "Mic" "Get" "Note"]
            addpath(mainDir)
            for i=subFolders
                if ~isfolder(mainDir+sessionFolder+filesep+i)
                    mkdir(mainDir+sessionFolder+filesep+i)
                    fprintf('Cartella Creata:\n in %s: %s\n',sessionFolder,i);
                end
                fprintf('\n');
            end
        end

        %% OutputFile - (file moving) Spostamento dei file.txt dalla ActualDir verso 'newDir/subfolder'
        function structBaseName = migraFile(structBaseName, newPath)
        %(?) cosa fa: migraFile(structBaseName, newPath) prende in input una
        %struct, dove ogni property al suo interno contiene una stringa con
        %il fullPath di tutti i files relativi a una misurazione conclusa, e,
        % sfruttando la funzione built-in di matlab 'movefile', sposta tali
        % files in una nuova directory newPath. Più precisamente, nella
        % nuova directory vengono generate delle sub-folders, ognuna avente
        % lo stesso nome della property di structBaseName. 
        %(*) nota: di solito (newPath) migraFile(..) si
        %propone come metodo di organizzazione e reindirizzamento dati,
        %molto comodo quando si generano in automatico molte simulazioni.
        %(^) Syntax: 
        % s = migraFile( s, 'C:\..newPath..\')
        %   (dove s contiene una property come [.Loa = 'C:\..myActualPath..\myFile.txt'])
        %   --> 'myFile.txt' viene spostato in:
        %   'C:\..newPath..\Loa\myFile.txt'
        %(=) output: la function restituisce la struct fornita in input,
        %con i percorsi aggiornati.

            for property=string(fieldnames(structBaseName))'

                fileName = strsplit(structBaseName.(property), filesep);
                fileName = string(fileName(end));
                newDest  = strjoin([newPath property fileName], filesep);
                movefile(structBaseName.(property), newDest);

                structBaseName.(property) = newDest;
            end
        end

        %% OutputFile - (switch off) Spegnimento Devices e chiusura dei file.txt
        function fileDest=deviceOff(channelValid, fileDest)
        %(?) cosa fa: dati in input un vettore di booleani (channelValid)
        %    che indica quali strumenti siano attualmente in uso per una
        %    misurazione (dq = cella di carico, dm = microfoni, dg = getti), e
        %    una struct (con le properties [.LC, .MI, .GE]) dei puntatori ai
        %    file in cui i dati vengono scritti, deviceOff si preoccupa di
        %    interrompere il processo di misurazione, stoppando l'esecuzione
        %    dei rispettivi dq, dm, e dg. Inoltre viene cancellata la cache di
        %    connessione, in modo da non entrare in conflitto con la prossima
        %    misurazione, che potrebbe leggere delle misurazioni latenti.
        %(*) nota: il canale dei getti viene stoppato e viene resettato
        %    alla tensione minima (5.4 Volt) per chiudere il getto d'aria.
        %(=) in uscita viene restituito (fileDest), solo che ora i
        %    puntatori ai file si riferiscono a file chiusi.

            global dq dm dg
            LibO = myOutputFile;
            for i=1:3
                if channelValid(i)
                    switch i
                        case 1
                            if LibO.isDevice(dq)
                            dq.stop; dq.flush;                            
                            end
                            fclose(fileDest.LC);
                        case 2
                            if LibO.isDevice(dm)
                            dm.stop; dm.flush;
                            end
                            fclose(fileDest.MI);
                        case 3
                            if LibO.isDevice(dg)
                            dg.stop;                       dg.flush;
                            dg.write( repmat(5.4, 2, 3) ); dg.flush;
                            end
                            fclose(fileDest.GE);
                    end
                end
            end
        end

        %% OutputFile - (DAQ connection check-up) Verifica dei devices DAQ in uso
        function esito=isDevice(dev)
        %(?) cosa fa: determina se un device è realmente connesso al
        %sistema DAQ, comparando la class dell'input con quella di default
        %per i sistemi hardware. Restituisce true/false.
        %(^) Syntax:
        %   esito = isDevice( dm|dq|dg )
           esito = isequal(class(dev),'daq.interfaces.DataAcquisition'); 
        end

        %% OutputFile - (file-pointer creation) Creazione automatizzata dei puntatori ai file Loa/Mic/Get
        function fileDest=creaFileTxt(channelValid, structFile)
          %(?) cosa fa: dato un vettore (channelValid) di booleani che
          %indichi quali tra i rispettivi canali Load|Mic|Get sono in
          %funzione per la simulazione, e una struct (structFile) con i percorsi dei
          %file adibiti alla scrittura, creaFileTxt si occupa della
          %creazione dei file e della formulazione degli headers in ognuno
          %di essi ("time" "Channel 1" "Channel 2" etc...)
          %(=) cosa restituisce: fornisce in output una struct con i
          %puntatori ai file appena creati, nelle rispettive properties (.LC, .MI, .GE)
          %(*) le properties in input e output sono volontariamente
          %differenti poiché quelle in input indicano i NOMI dei file che
          %si intende creare, mentre in output i PUNTATORI dei file appena
          %creati.
          %(^) Syntax:
          %  per creare tutti e 3 i files:
          %     f = creaFileTxt( [1 1 1], s),   dove s possiede [.Loa, .Mic, .Get]

          fileDest = struct;
          LibO = myOutputFile;
          for i=1:3
            if channelValid(i)
             switch i
              case 1
                fileDest.LC=LibO.preparaFile( structFile.Loa, ["Time" "LoadCell" "Unused" "Pitot" "Contraction"]);
              case 2
                fileDest.MI=LibO.preparaFile( structFile.Mic,["Time" "Mic"+(1:16)]);
              case 3
                fileDest.GE=LibO.preparaFile( structFile.Get,["Time" "TOP" "BOT" "SIDE"]); %<< modificato 5 Aprile
             end
            end
          end
        end
         
       %% OutputFile - (file list retrieving) Ricerca di tutti i file associati ad una simulazione
       function s = txt2struct(fileName,parentFolder,msg,askForPlot)
           %(?) cosa fa: questa function è molto utile poiché permette di
           %collezionare tutte i file *.txt associati ad una simulazione a
           %partire da un solo file, risparmiando così il tempo di cercare manualmente
           % tutte le informazioni a catalogo. In più è possibile
           % richiedere una rapresentazione dei risultati.
           %
           %(*) per "file correlati" si fa riferimento a tutti i file che
           %condividono lo stesso timestamp (data ora).
           %
           %(=) cosa restituisce: restituisce una struct 's' con queste
           %properties: 
           % s.loa = il fullPath delle misurazioni di Load Cell (se presente )
           % s.get = il fullPath delle misurazioni dei Getti    (se presente )
           % s.mics = il fullPath delle misurazioni dei Microfoni (se presente )
           % s.note = il fullPath con i logs informativi della simulazione.
           % s.dev  = ["get" "loa" "mics" "note"] un vettore di keywords in
           %          base al numero di file correlati trovati. (se ad esempio non
           %          si trova il file relativo ai microfoni, 'mics' non è nel vettore).
           % s.valid = [1 1 1] un vettoredi true/false che dice quali
           %          dispositivi hardware (loa|mic|get) sono stati
           %          utilizzati.
           %
           %(>) input list:
           %    1- fileName : stringa rappresentante uno dei file della
           %      simulazione desiderata (full path). Se fileName=[] ->
           %      Matlab apre una finestra per cercare manualmente il file.
           %    2- parentFolder : (facoltativo!) stringa con il nome della cartella in cui
           %      cercare gli altri file correlati (NO full path).  Se
           %      questo input è vuoto, il parentFolder viene desunto dal
           %      fullPath immesso con il primo input (fileName).
           %    3- msg : (facoltativo!) una stringa di supporto da
           %      riprodurre sulla finestra di esplorazione file per
           %      guidare l'utente nella sua scelta, specie utilizzato se
           %      il primo input è vuoto.
           %    4- askForPlot : (facoltativo!) keyword stringa ("OneFigure"|"Multiple")
           %      se presente, plotta i risultati. <Consulta la function
           %      'plotMySym'>
           %
           %(^) Syntax:
           %  s = txt2struct([])
           %  s = txt2struct( 'C:\..fullPath..\Loa\myTEST_Loa.txt' )
           %  s = txt2struct([], 'myFolder')
           %  s = txt2struct([], [], 'messaggio Extra per Utente')
           %  s = txt2struct([], [], [], 'oneFigure' )

           if (nargin<3) || isempty(msg)
               if (nargin>1) && ~isempty(parentFolder)
                 msg = "dentro ("+parentFolder+")";
               else
                 msg = ""; 
               end
           end

           if isempty(fileName)
               a=0;
               while ~a
                   [a,b]=uigetfile('*.txt',"Scegli un file *.txt da leggere "+msg);
               end
               fileName = [b a];
           end
           if (nargin == 1) || isempty(parentFolder)
               depthFolder = 2; 
               parentFolder=strsplit(fileName,filesep);
               parentFolder=parentFolder{end-depthFolder};
           end

           timestamp = regexp(fileName,'\(.*\)','match');

           if isempty(timestamp)
               disp('questo file non ha una data!?')
               s=struct();
           else
               timestamp = timestamp{1};
               disp("cerco tutti i files etichettati: """+timestamp+"""")

               mainP = strsplit( fileName,filesep);
               idx   = find( strcmp(mainP, parentFolder),1,'last');
               searchStr = strjoin( ""+mainP(1:idx), filesep);
               searchStr = searchStr+repmat([filesep '*'],1,length(mainP)-1-idx);

               %attraverso il comando matlab
               %"dir('C:\..fullPath..\parentFolder\*\*(timestamp).txt'
               %posso trovare tutti i file con lo stesso timestamp in tutte
               %le subforlders. (se ci sono più livelli di subfolder,
               %bisogna solo regolare il numero di '\*' nella searchString).
               searchStr = searchStr+filesep+("*"+timestamp+"*.txt");
               correlati = dir(searchStr);

               s=struct('dev',repmat("",0,1));
               words = ["loa" "mics" "get" "note"];
               for i=1:length(correlati)
                nome = correlati(i).name;
                path = correlati(i).folder;
                tag  = replace(regexp(nome,'_[A-z]{2}','match'),'_','');
                tipo = validatestring(tag{1},words);
                s.(tipo)=[path filesep nome];
                s.dev(end+1) = tipo;

               % % sarebbe potenzialmente pericoloso importare tutti i dati.
               %temp = importdata([path filesep nome]);
               %s.(tipo) = array2table(temp.data,'VariableN',temp.colheaders); 
               end
               s.valid = ismember(words(1:3),string(fieldnames(s)'));
           end

           if nargin==4
             myOutputFile.plotMySym(s,askForPlot); 
           end
       end

       %% OutputFile - (closed simulation plot) Rappresentazione grafica dei risultati di una simulazione passata
       function plotMySym(s,figureMode,extras)
           %(?) cosa fa: se la function di libreria extractFromFile
           %semplicemente estrae il contenuto di un qualunque file
           %(loa/mics/ get), è la function plotMySym(s, figureMode) a
           %rappresentare il contenuto della simulazione in tutti i suoi
           %aspetti.
           %(>) input list:
           %  1- s : una struct con properties associate ai nomi dei file
           %    cui la simulazione fa riferimento.
           %  2- figureMode: (input facoltativo) una keyword stringa, a scelta tra "OneFigure"
           %    (per visualizzare tutti i dati in un'unica figure con assi separati. default),
           %     e "MultipleFig" per visualizzare Getti, Microfoni, e cella
           %     di Carico & Pitot separatamente.
           %  3- extras: (input facoltativo) una lista di keyword
           %     aggiuntive da includere nei plot, ["cd" "drag" "xcorr"]
           %(^) Syntax:
           %  s = txt2Struct( [], 'myCartellaTest') <-- sceglie la simulazione 
           %  plotMySym(s, "OneFigure")    <vale anche una abbreviazione, es. "OneF">
           %  plotMySym(s, "MultipleFig")  <vale anche una abbreviazione, es. "Mult">
           %(*) su ogni plot viene presentata una legenda, se si clicka sui
           %relativi segnali nella legenda, si può impostare la visibilità
           %del plot tra on e off.

           nPlot = sum(s.valid);  %<-- conta il numero di tipologie di segnali (massimo 3, quando la simulazione ha registrato sia Load, Mics e Get)
           % servirà per capire come suddividere i subplot nel caso tutto
           % venga rappresentato in un'unica figure.
           if (nargin > 1)
              figureMode = validatestring(figureMode,["OneFigure" "MultipleFig"]);
           else
              figureMode = "OneFigure";
           end

           if isequal(figureMode,"OneFigure")

            switch nPlot
               case {1,2,3}
               gridP = [nPlot 1];
               case 4
               gridP = [2 2];
            end
           else
              gridP = [1 1]; 
           end

           k=0;
           for i=s.dev
            if ~strcmp(i,"note") 
              k=k+1;
              setBaseAx() %"setBaseAx" è definita in fondo alla function.
                          % e inizializza la figure connotandola con un
                          % nome.
              [data, t, tags]=myOutputFile.extractFromFile(s.(i));

              if ~iscell(data)

              switch i
                 case 'loa'
                  plot(t,data,'-'), hold on
                  ll=legend('VariableNames',tags+" [V]");
                  ylabel('Device Signals')
                  
                  if nargin>2
                     LibT = myTdmsReader;
                     Zero = [2.872748438 -0.088105225];
                     out = LibT.computeCd( data(:,[1 4]), [], Zero);
                     
                     if any(contains(lower(extras),"cd"))
                      % plot the cd coeff (<!> trascurando la atmosfera del file)
                      plot(t, out.Cd, '-','DisplayN','Cd');
                     end
                     if any(contains(lower(extras),"drag"))
                      % plot the drag (<!> trascurando la atmosfera del file)
                      plot(t, out.Drag, '-','DisplayN','Drag [N]');
                     end 
                     if any(contains(lower(extras),"u"))
                      % plot the U.inf (<!> trascurando la atmosfera del file)
                      plot(t, out.Uinf / 10, '-','DisplayN','U_{\infty}/10 [m/s]');
                     end
                     ylim auto
                     
                    % questa sezione scrive i valori medi del caso con 5
                    % velocità del vento
%                      for ii=1:5
%                          
%                          ok = (t<=60*ii-5) & (t>=60*(ii-1)+5);
%                          cdm(ii) = mean( out.Cd(ok) );
%                          uim(ii) = mean( out.Uinf(ok) );
%                          x=30+(ii-1)*60;
%                          text( x, uim(ii)/10+0.1, ...
%                               sprintf('U: %2.2f [m/s]',uim(ii)),'hor','center')
%                           
%                          text( x, uim(ii)/10-0.1, ...
%                               [sprintf('Re: %1.2f ',uim(ii)*1.225*0.17/1.5) '\cdot 10^5'],'hor','center')
%                          
%                          %cdL=[0.3144	0.3810	0.4292	0.4654	0.4941];
%                          text( x, cdm(ii)-0.1, sprintf('Cd: %1.3f',cdm(ii)),'hor','center');
%                      end
%                      cdm, uim
                  end
                   
                 case 'get'
                  plot(t,data,'-'), hold on
                  ll=legend('VariableNames',tags);
                  ylim([5.4 9.6]), yticks(5.5:9.5) 
                  ylabel('Jet signals [V]')
                 case 'mics'
                     ss=size(data,2);
                     for j=1:ss
                     plot3(t,repmat(j,size(t)),data(:,j),'-'), hold on
                     end
                  ll=legend('VariableNames',tags);
                  ll.NumColumns = 2;
                  ylim([0 ss+1]), yticks([])
                  ylabel('Channels')
                  zlabel('mic pressure')
                  
                  %[6,9,16]
                  if nargin > 2
                     if any(contains(lower(extras),'xcorr'))
                     % [c,lags]=xcorr()
                     end
                  end
              end
              xlim([0 t(end)])
              xlabel('time [s]')
              ll.ItemHitFcn = @myTdmsReader.ShowOnOff; 
              else
               title({'File Vuoto o Illeggibile:'; 
                    replace(s.(i),{filesep, '_'},{'\\','\_' })});
              end
            end
           end

           function setBaseAx()
               if (k==1) || ~isequal(figureMode,"OneFigure")
                   nome = strsplit(s.(i),filesep);
                   figure('Name',nome{end},'Color','w');
               end
               if isequal(figureMode,"OneFigure")
                   subplot(gridP(1),gridP(2),k);
               end
           end
       end

       %% OutputFile - (file-data extraction) Estrazione dati da un file *.txt
       function [data, t, tags]=extractFromFile(txtFileName)
       %(?) cosa fa: questa function prende in input il
       %nome di un file in formato *.txt (percorso completo), e
       %legge le informazioni ivi contenute riguardo la simulazione.
       %(=) leggendolo, estrae il suo contenuto e riporta questi output:
       %  1- data  : una matrice [N x M] dove N sono gli scans e M il numero dei canali
       %  2- t     : il vettore temporale [N x 1]. Se non è presente la
       %             colonna con header 'time' allora il vettore rappresenta i samples da 1 a N 
       %  3- tags  : un vettore di stringhe con tutti gli header (escluso "time")
       %             associati ad ognuno degli M canali.
       %(*) se il file è vuoto o illeggibile, allora t=[], tags=[], data={error MSG}
       %(^) Syntax:
       %   [data, t, tags] = extractFromFile( 'C:\ fullPath ..\myTest_Loa.txt')

           content = importdata(txtFileName);
           if isstruct(content)
               tags = content.colheaders;
               data = content.data;
               try       
                   validatestring('time',tags); %<-- se va in errore si attiva il catch!
                   
                   tags(1) = [];  %elimino 'time' dalla lista headers.
                   if isequal(tags, {'TOP','SIDE','BOT'})
                       tags(2:3) = tags([3 2]);
                   end
                   t = data(:,1);
                   data(:,1) = [];
               catch
                   disp('no time vec')
                   t = 1:size(data,1);
               end
           else
               data = {'File Vuoto o Illeggibile:'; txtFileName}; 
               disp(data);
               t=[]; tags=[];
           end
       end

       %% OutputFile - (simulation offline-replay) Riproduzione di una simulazione conclusa in assenza di connessione DAQ 
       function ReviveSym(s,mode,delTime)
       % (?) cosa fa: questa function ha come scopo quello di riaprire l'interfaccia dopo
       % che l'utente è tornato a casa, lontano dal sistema DAQ,
       % riproducendo esattamente ciò che l'utente ha visto
       % sull'interfaccia in laboratorio, una sorta di "replay", magari per fare qualche screenshot.
       % (>) input list:
       %   1- 's' è una struct con i nomi dei file a cui la simulazione che l'utente vuole
       %      visualizzare si riferisce, può possedere una o più delle seguenti
       %      properties: [.loa, .mics, .get, .note]
       %   2- 'mode' è una stringa keyword. Se l'utente sceglie "completed"
       %      è perché vuole vedere l'interfaccia nel suo stato finale, a
       %      simulazione conclusa. Se l'utente sceglie "emulate" invece
       %      l'interfaccia riproduce un timelapse della simulazione in
       %      esame (non importa la durata effettiva) in 1000 frames.
       %   3- 'delTime' (facoltativo) è un numero, inferiore alla durata di
       %      simulazione. Se non presente, delTime = durata/2.
       % (^) Syntaxes:
       %    s = txt2struct([],"myCartellaTest")  
       %    ReviveSym( s, "emulate")    <va bene anche una versione contratta come 'emul'>  
       %    ReviveSym( s, "completed")  <va bene anche una versione contr atta, come 'comp' o 'complete'
       %    ReviveSym( s, "emulate",15) <applica la suddivisione a 15 [sec]
           %determino la modalità di funzionamento.
           tipi = ["completed" "emulate"];
           if nargin>1
           mode = validatestring(mode, tipi);
           else
           mode = tipi(1);
           end
           
           %richiamo librerie ausiliarie
           LibO = myOutputFile;
           LibI = myInterfacciaDAQ;

           %crea l'interfaccia da capo e i function handle di rendering.
           if ~exist('ax','var') || ~exist('PL','var')     
              global ax PL info %#ok<TLEV>
              
              [~,ax,PL,info]=LibI.creaInterfaccia();        
              [PL.Mic, updMic, ~]=LibI.preparaRear(ax{4});
              [PL.SurfM,    updMic.surf]=LibI.addSurf(PL.Mic.XData, PL.Mic.YData, 100);
               
              % riporto il nome del file della simulazione in cima
              % all'interfaccia.
              ss=strsplit(s.loa,filesep);
              ax{1}.Parent.Name = ss{end};
              ax{4}.Title.String = 'Simulation Replay';
           end

           %si informa di quali canali la simulazione ha usufruito.
           channelValid= s.valid;
           tipi = ["loa" "mics" "get"];

           %estrae i numeri da ogni file (load, mic, get) che riesce a trovare di quella
           %simulazione. poi passa alla loro rappresentazione nel dominio temporale: questo
           %equivale ad eseguire di default la modalità 'completed'.
           nums=cell(1,3); t=cell(1,3);
           for i=1:3
               if isfield(s, tipi(i))
                   
                   % Estraggo i dati di tutta la registrazione e i relativi tempi
                   [nums{i}, t{i},~]=LibO.extractFromFile(s.(tipi(i)));
                   
                   if ~iscell(nums{i})
                   switch i
                       case 1 
                           % nel caso della cella di carico, resetta il
                           % plot e poi displaya il canale.
                           durata = t{i}(end)+t{i}(2);
                           LibI.resetPlots(durata,channelValid)
                           LibI.updLC(t{i}, nums{i});

                       case 2
                           % nel caso dei mics, imposta una visualizzazione
                           % 2D, con superficie visibile, applicando l'rms
                           % (root mean square).
                           go3D = false;
                           LibI.updMI(nums{i}, go3D, updMic.surf, @rms);
                       case 3
                           % nel caso dei getti, recupera la frequenza di
                           % trasmissione contando il numero di scan
                           % presenti in 1 secondo.
                           channelRate = nnz(t{i}<1);

                           % poi plotta i getti, e poi linka la vista delle
                           % ascisse dei 3 assi.
                           LibI.plotJet( ax(1:3), nums{i}, channelRate, channelValid(3));
                           linkprop([ax{1:3}],'XLim');
                   end
                   end
               end
           end

           % se è stato recuperato anche il file di Note della simulazione,
           % mette a display i logs.
           if any(strcmp("note",fieldnames(s)))
               infos = flipud(importdata(s.note,'s'));
               infos = cellfun(@(x) replace(x,'= nan','= ...'),infos,'UniformOutput',false);
               fn=fieldnames(info);
               arrayfun(@(i) set(info.(fn{i}),'String',infos{i}),1:length(fn));

               PL = info.di.UserData( info.di, PL);
           end

           % se sono in modalità 'emulate', dopo che l'interfaccia ha
           % precedentemente rappresentato i risultati, ora si limita a
           % focalizzare la visuale di ogni grafico solamente alla finestra
           % temporale dedicata, simulando così un comportamento real-time.
           if strcmp(mode,'emulate')
               linkprop([ax{1:3}],'XLim');
               minDuration = min( nonzeros( cellfun(@length, t)));
              idxVec = round(linspace(1, minDuration,300));  %<< scompongo la timeline del Test
              lastIdx = 1;
              
              for k=idxVec(2:end)
                  time = t{1}(k);
                  ax{5}.XLim = [0 time+0.01];
                  ax{1}.XLim = time+[-1 1]*0.5;
                  LibI.updMI(nums{2}(lastIdx:k,:), go3D, updMic.surf);
                  caxis(ax{4},[0 max(nums{2},[],'all')])
                  for i=1:3
                      PL.LS(i).XData = time*[1 1];
                  end
                  drawnow
              end
           end
            
           if nargin<3
           delTime = durata/2; % se non si fornisce il delay-time si ipotizza che la simulazione sia divisa equamente!
           end
           
           PL.recap(1).XData = [0 delTime];
           PL.recap(2).XData = [delTime durata];
           PL.recap(1).YData = [1 1].*mean( nums{1}( t{1}<delTime,1));
           PL.recap(2).YData = [1 1].*mean( nums{1}( t{1}>delTime,1));
           PL.CO.Visible = 'off';
           
           try
               cd  = regexp(info.cd.String,'[\.\d]{2,}','match'); 
               cd0 = regexp(info.c0.String,'[\.\d]{2,}','match');
               
               cd = str2num(cd{1}); %#ok<*ST2NM> % non sostituire str2num con str2double!!!
               cd0 = str2num(cd0{1});
               
               dcd = (cd-cd0)/cd0;
           ax{5}.Title.String = sprintf('\\DeltaCD = %+.1f%% (%+.4f)', dcd*100,cd-cd0);
           set(ax{5}.Children(1:2),'LineWidth',2)
           ax{5}.XLabel.String = 'time [s]';
           catch ME, disp(ME)
           end
       end

       %% OutputFile - (XML backup) Creazione di un file *.txt di backup dei risultati di simulazione in formato XML
       function send2backup( traccia, fileN, tempo, meanLC, deltaCD, CD0, CD)
       % (!) premessa: dal momento che non tutti i professori possiedono matlab
       % sul pc, è stata pensata anche una modalità di visualizzazione dei
       % test eseguiti in laboratorio con annesse performance esterna al
       % linguaggio matlab. 
       % (?) cosa fa: send2backup prende in ingresso una lunga lista di
       % input per generare un file *.txt con un testo codificato in XML,
       % il quale può essere letto da un particolare file (results.html)
       % di facile utilizzo, su un qualunque browser, come google chrome.
       % In sostanza, ogni test svolto DURANTE un processo di
       % ottimizzazione (eseguito cioè in automatico) ottiene una sua copia
       % backup con codifica XML.
       % (*) il nome del file di testo è lo stesso della sessione di test,
       % ovvero [parentFolder].txt
       % (>) input list:
       %  1- traccia: un cell array {1 x 3} con vettori che contengano la
       %    porzione pseudo-periodica dei segnali Getto per top/side/bot.
       %  2- fileN:   la stringa con il nome del file di salvataggio (percorso completo)
       %  3- tempo:   scalare, durata della simulazione [s]
       %  4- meanLC, deltaCD, CD0, CD  : 4 scalari rappresentanti il valor
       %  medio di tensione della cella di carico, il delta CD, e i
       %  coefficienti di resistenza del caso non forzato (getti spenti)
       %  CD0, e del caso forzato (getti accesi) CD.

           global parentFolder
           file = parentFolder+".txt";
           f=fopen(file,'a');
           data = regexp(fileN,'\(.*\)','match')+" dCD: "+deltaCD;
           fprintf(f,"\n<button class='collapsible'>%s</button>",data{1});
           fprintf(f,"\n<div class='content'>\n<ul>");
           fprintf(f,"\n\t<li>%s</li>", "<b>local file:</b> "+fileN);
           fprintf(f,"\n\t<li>%s</li>", "<b>durata:</b> "+tempo+" [s]");
           fprintf(f,"\n\t<li>%s</li>", "<b>[CD -CD0]:</b> "+deltaCD);
           fprintf(f,"\n\t<li>%s</li>", "<b>CD0:</b> "+CD0);
           fprintf(f,"\n\t<li>%s</li>", "<b>CD:</b> "+CD);
           if isnan(meanLC)
               meanLC = 'nan';
           end
           fprintf(f,"\n\t<li>%s</li>", "<b>mean LC: </b> "+meanLC+" [V]");

           canali = ["top" "bot" "side"]; %< modificato il 5 Aprile 
           for i=1:3
              d = strjoin(traccia{i}+"",';');
              fprintf(f,"\n\t<li><b>"+canali(i)+":</b> <p class='vet' id='%s'>\n\t[%s]\n\t</p></li>",canali(i),d);    
           end

           fprintf(f,"\n</ul>\n</div>");
           fclose(f);
       end

       %% OutoputFile - (GA overview) Genetic Algorithm Fitness Function performance visualization
       function ax=GArecap(history, otherVars, ~)
       %(?) cosa fa: GArecap(history) prende in input una struct e mostra su
       %figures esterne a quelle dell'interfaccia le statistiche e le
       %performance del processo di ottimizzazione del GA appena eseguito.

           figure

           bests = cat(1, history(:).best);
           [superBest, idx] = min( cat(1,history(:).score));
           ss = [size(history(1).pop,1) length(history)];
           [idxI, idxG] = ind2sub( ss, idx);
           disp('Champion Optimization History:')
           disp("Gen "+idxG+" Test n° "+idxI)
           disp('Values')
           disp(history(idxG).pop(idxI,:)')
           plot(1:length(bests), bests, ':b'); hold on
           plot(1:length(bests), bests, '.k','LineW',2); hold on
           plot(idxG, superBest, 'pr');
           ax=gca; makeTitle = @(str) set(ax.Title,"String",str);
           for i=1:ss(2)
               color = rand(1,3);
               for j=1:ss(1)
                   stringa = "x: [" +strjoin(history(i).pop(j,:)+"",", ")+"]";

                   ud.xs = stringa;
                   ud.x  = history(i).pop(j,:);
                   ud.pop = [i j];
                   ud.score = history(i).score(j);
                   fullstr = sprintf('\\{Gen:%d, Item:%d, Score:%.2e\\}    %s',ud.pop(1), ud.pop(2), ud.score, ud.xs);
                   ud.fs = fullstr;
                   plot(i,history(i).score(j),'o','Color',color,'UserData',ud,...
                       'ButtonDownFcn',@(src,evt) makeTitle(src.UserData.fs));
               end
           end
          
           ax.ButtonDownFcn = @(src,evt) makeTitle("GA history");
           title('GA history')
           l1 = plot(nan,nan,':b','DisplayName','Line of Minimums');
           l2 = plot(nan,nan,'.k','DisplayName','Best fitness of Generation');
           l3 = plot(nan,nan,'o','Color',[1 1 1]*.5,'DisplayName','Generic test');
           l4 = plot(nan,nan,'pr','DisplayName',sprintf('Minimum found: Gen.%d Item:%d',idxG, idxI));

           legend([l1 l2 l3 l4]), ax.Parent.Color = [1 1 1];
           ylabel(["fitness fcn"; "CD-CD_0"]), set(gcf,'Color','w')
           xlabel('generations'), xticks(1:length(history)), xlim([0.5 length(history)+0.5])
           
           % {figure1, figure2, figure3} --> in ognuno: [righe x colonne]
           if nargin > 1
               hasConv = isfield(history,'cc') && isstruct( history(1).cc);
               if hasConv
                  cc = history(1).cc;
                  LB = cc.LB;
                  UB = cc.UB; 
                  classNames = cc.classes(repelem( cc.classID, [cc.quantity{:}]));
               end
               idxs = arrayfun( @(h) find( h.score == min(h.score),1), history);
               for figs=1:length(otherVars)
                   figure('Color','w'), hold on;
                   k = 1; vv = otherVars{figs}; 
                   for v = vv(:)'
                    subplot(size(vv,2), size(vv,1), k);
                    plot( 1:ss(2), arrayfun(@(x) history(x).pop(idxs(x),v),1:ss(2)) ,':b')
                    hold on
                    for i = 1:ss(2)
                      plot(repmat(i, ss(1), 1), history(i).pop(:,v),'o'); hold on
                      plot(  i,  history(i).pop( idxs(i), v), '.k');
                    end
                    xticks(1:ss(2));
                    if hasConv
                     ylabel("x_{"+v+"}: "+string(classNames(v)))
                     ylim([LB(v) UB(v)])
                    else
                     ylabel("x_{"+v+"}")
                    end
                    k=k+1;
                   end
                   xlabel('Generations')
               end
           end
           
           if nargin > 2
             if hasConv
               figure('Color','w')
               
               poA = polaraxes();
               sc = size(history(1).pop,2);
               theta = linspace(0,2*pi,sc+1);
               theta(end)=[];
               poA.ThetaAxis.TickValues = rad2deg(theta);
               
               poA.ThetaAxis.TickLabels="(x_{"+(1:sc)+"}) "+string(classNames);
                faA = axes(); faA.Position = poA.Position;
              for i=1:ss(2)  
                for j=1:ss(1)
                 yy = history(i).pop(j,[1:end 1]);
                 [xx,yy] = pol2cart(theta([1:end 1]),(yy-LB([1:end 1]))./(UB([1:end 1])-LB([1:end 1]))); 
                 plot(xx,yy,'.','LineW',0.01,'Color',[1 1 1]/10,'Visible','on'), hold on
                end
              end
               xlim(faA,[-max(get(poA,'RLim')),max(get(poA,'RLim'))]); 
               ylim(faA,[-max(get(poA,'RLim')),max(get(poA,'RLim'))]);
               axis square
               faA.Toolbar.Visible = 'off';
               faA.Visible = 'off';
               
               jj = jet; jj=jj(round(linspace(1,length(jj),ss(2))),:);
               for i=1:ss(2)
                 yy = history(i).pop(idxs(i),:);
                 [xx,yy] = pol2cart(theta,(yy-LB)./(UB-LB));
                 fill( xx, yy,jj(i),'EdgeColor','k','FaceA',0.1);
                 hold on
                 drawnow, pause(1)
               end
               yy = history(idxG).pop(idxI,[1:end 1]);
               [xx,yy] = pol2cart(theta([1:end 1]),(yy-LB([1:end 1]))./(UB([1:end 1])-LB([1:end 1]))); 
               plot(xx,yy,'-','Color',[1 0 0],'LineW',2)
               
               
             end
           end
       end
       
    end
end

%%